#include <Blobs/StdV3D.h>
#include <assert.h>


#include "BoundObb.h"

namespace V3D {

////////////////////////////////////////////////////////////////////////////////

BoundObb::BoundObb( const Vector3D& vecInitPoint )
: m_vcHalfSize( 1E-5f, 1E-5f, 1E-5f )
{
	m_matCoordSystem.LoadIdentity();
	m_matCoordSystem.SetTranslation( vecInitPoint );
}



				//////////////////////////////////////////////

void BoundObb::Build(const Vect3DArray& aVertices, const TriangleArray& aTriangles)
{
	// Calcul des vertices utilises par les triangles 
	Vect3DArray::size_type   nVertexCount = aVertices.size();

	Vect3DArray	aPointCloud;
	aPointCloud.reserve( nVertexCount);
	
	std::vector<bool> abUsedVertex( nVertexCount, false);

	typedef TriangleArray::const_iterator TriIter;
	TriIter itEnd = aTriangles.end();
	for( TriIter it = aTriangles.begin(); it != itEnd; ++it)
	{
		abUsedVertex[ it->a ] = true;
		abUsedVertex[ it->b ] = true;
		abUsedVertex[ it->c ] = true;
	}

	for( Vect3DArray::size_type i = 0; i < nVertexCount; ++i)
	{
		if( abUsedVertex[i] )
			aPointCloud.push_back( aVertices[i] );
	}

	Build( aPointCloud);
}	
	

				//////////////////////////////////////////////


void BoundObb::Build(const Vect3DArray& aPointCloud)
{
	// Nuage vide ?
	int32 nPointsCount = aPointCloud.size();

	if( nPointsCount == 0)
	{
		m_matCoordSystem.LoadIdentity();
		m_vcHalfSize.Set( 0.f, 0.f, 0.f);
		return;	
	}

	// Calcul du centre de gravite du nuage de points
	double dPointsCountInv = 1.0 / (double) nPointsCount;

	double dAvgX = 0;
	double dAvgY = 0;
	double dAvgZ = 0;
	
	for( int32 i = 0; i < nPointsCount; ++i)
	{
		const Vector3D& curVect = aPointCloud[i];
		assert( dAvgX == dAvgX );
		dAvgX += curVect.x;
		dAvgY += curVect.y;
		dAvgZ += curVect.z;
	}
	
	dAvgX *= dPointsCountInv;
	dAvgY *= dPointsCountInv;
	dAvgZ *= dPointsCountInv;
	
	Vector3D vcAvg( (float)dAvgX, (float)dAvgY, (float)dAvgZ);

	// Calcul de la matrice de covariance du nuage de points
	double dCovXX = 0.f;
	double dCovXY = 0.f;
	double dCovXZ = 0.f;
	double dCovYY = 0.f;
	double dCovYZ = 0.f;
	double dCovZZ = 0.f;
	
	for( int32 i = 0; i < nPointsCount; ++i)
	{
		Vector3D curVec = aPointCloud[i]; curVec -= vcAvg;
		dCovXX += curVec.x * curVec.x;
		dCovXY += curVec.x * curVec.y;
		dCovXZ += curVec.x * curVec.z;
		dCovYY += curVec.y * curVec.y;
		dCovYZ += curVec.y * curVec.z;
		dCovZZ += curVec.z * curVec.z;
	}

	Vector3f vecCovX = Vector3f( float(dCovXX), float(dCovXY), float(dCovXZ) );
	Vector3f vecCovY = Vector3f( float(dCovXY), float(dCovYY), float(dCovYZ) );
	Vector3f vecCovZ = Vector3f( float(dCovXZ), float(dCovYZ), float(dCovZZ) );

	// Arrondit  0 les valeurs proches de 0
	// Cela amliore la qualit de la bounding box
	MatrixTransf matCovariance( vecCovX, vecCovY, vecCovZ);
	
	matCovariance.RoundToZero();
	
	// Calcul des vecteurs propres de la matrice de covariance. Ce seront les axes de l'OBB.
	Vector3f vcObbDirX, vcObbDirY, vcObbDirZ;
	matCovariance.ComputeSymmetricMatrixEigenVectors( vcObbDirX, vcObbDirY, vcObbDirZ);

	vcObbDirX.Normalize();
	vcObbDirY.Normalize();
	vcObbDirZ.Normalize();


	assert( fabsf(vcObbDirX * vcObbDirY) < 1E-5f );
	assert( fabsf(vcObbDirX * vcObbDirZ) < 1E-5f );
	assert( fabsf(vcObbDirY * vcObbDirZ) < 1E-5f );

	// Calcul des dimensions de la boite, et determination de son centre
	Vector3f vec0 = Vector3f::ConvertFrom( aPointCloud[0] - vcAvg );

	Vector3f vcMinAABB =vec0;
	Vector3f vcMaxAABB = vcMinAABB;

	Vector3f vcMinOBB( vec0 * vcObbDirX, vec0 * vcObbDirY, vec0 * vcObbDirZ);
	Vector3f vcMaxOBB = vcMinOBB;

	for( int32 i = 1; i < nPointsCount; ++i)
	{
		Vector3D curPtd = aPointCloud[i]; curPtd -= vcAvg;
		Vector3f curPt = Vector3f::ConvertFrom( curPtd );

		if( vcMinAABB.x > curPt.x) vcMinAABB.x = curPt.x;
		if( vcMinAABB.y > curPt.y) vcMinAABB.y = curPt.y;
		if( vcMinAABB.z > curPt.z) vcMinAABB.z = curPt.z;
		if( vcMaxAABB.x < curPt.x) vcMaxAABB.x = curPt.x;
		if( vcMaxAABB.y < curPt.y) vcMaxAABB.y = curPt.y;
		if( vcMaxAABB.z < curPt.z) vcMaxAABB.z = curPt.z;

		Vector3f vcCurOBB( curPt * vcObbDirX, curPt * vcObbDirY, curPt * vcObbDirZ);
		if( vcMinOBB.x > vcCurOBB.x ) vcMinOBB.x = vcCurOBB.x;
		if( vcMinOBB.y > vcCurOBB.y ) vcMinOBB.y = vcCurOBB.y;
		if( vcMinOBB.z > vcCurOBB.z ) vcMinOBB.z = vcCurOBB.z;
		if( vcMaxOBB.x < vcCurOBB.x ) vcMaxOBB.x = vcCurOBB.x;
		if( vcMaxOBB.y < vcCurOBB.y ) vcMaxOBB.y = vcCurOBB.y;
		if( vcMaxOBB.z < vcCurOBB.z ) vcMaxOBB.z = vcCurOBB.z;
	}

	Vector3D vcSizeAABB; vcSizeAABB.Set(vcMaxAABB - vcMinAABB);
	const LengthType lVolumeAABB = vcSizeAABB.x * vcSizeAABB.y * vcSizeAABB.z;
	
	Vector3D vcSizeOBB; vcSizeOBB.Set(vcMaxOBB - vcMinOBB);
	const LengthType lVolumeOBB = vcSizeOBB.x * vcSizeOBB.y * vcSizeOBB.z;
	
	// Decision si on garde l'AABB ou l'OBB (on garde la moins volumineuse).
	// Le cas ou on garde l'AABB ne devrait arriver que tres rarement.
	// vcOffsetCenter = Position, dans le repere de l'OBB, du centre de l'OBB par rapport au centre de gravite
	Vector3D vcSize = vcSizeOBB;
	Vector3D vcOffsetCenter; vcOffsetCenter.Set( (vcMaxOBB + vcMinOBB) * 0.5f );

	if( lVolumeOBB > lVolumeAABB * 1.001)
	{
		vcSize = vcSizeAABB;
		vcOffsetCenter.Set( (vcMaxAABB + vcMinAABB) * 0.5f );
		vcObbDirX.Set(1.f, 0.f, 0.f);
		vcObbDirY.Set(0.f, 1.f, 0.f);
		vcObbDirZ.Set(0.f, 0.f, 1.f);
	}

	m_vcHalfSize = vcSize * 0.5f;
	// Donner une epaisseur aux objets completement plats
	m_vcHalfSize += Vector3D( 1E-5f, 1E-5f, 1E-5f );

	assert( m_vcHalfSize.x > 0 && m_vcHalfSize.y > 0 && m_vcHalfSize.z > 0 );


	// On dtermine enfin la position du centre de l'OBB
	
	m_matCoordSystem.SetColumns( vcObbDirX, vcObbDirY, vcObbDirZ);
	m_matCoordSystem.SetTranslation(0, 0, 0);
	
	Vector3D vcCenter = vcAvg + (m_matCoordSystem * vcOffsetCenter);
	m_matCoordSystem.SetTranslation( vcCenter);
}



				//////////////////////////////////////////////

void BoundObb::ComputeApproxBoundSphere( Vector3D& vcCenter, float& fRadius) const
{
	fRadius  = float( m_vcHalfSize.GetLength() );
	vcCenter.Set( m_matCoordSystem.GetTranslation() );
}

				//////////////////////////////////////////////

void BoundObb::GetCorners( Vector3D aObbVerts[8]) const
{
	Vector3f vecXf, vecYf, vecZf;
	Vector3D vcCenter;

	vcCenter.Set(m_matCoordSystem.GetTranslation());
	m_matCoordSystem.GetColumns(vecXf, vecYf, vecZf);

	Vector3D vecX; vecX.Set( vecXf );
	Vector3D vecY; vecY.Set( vecYf );
	Vector3D vecZ; vecZ.Set( vecZf );

	vecX *= m_vcHalfSize.x;
	vecY *= m_vcHalfSize.y;
	vecZ *= m_vcHalfSize.z;


	aObbVerts[0] = vcCenter - vecX - vecY - vecZ;
	aObbVerts[1] = vcCenter + vecX - vecY - vecZ;
	aObbVerts[2] = vcCenter + vecX + vecY - vecZ;
	aObbVerts[3] = vcCenter - vecX + vecY - vecZ;
	aObbVerts[4] = vcCenter - vecX - vecY + vecZ;
	aObbVerts[5] = vcCenter + vecX - vecY + vecZ;
	aObbVerts[6] = vcCenter + vecX + vecY + vecZ;
	aObbVerts[7] = vcCenter - vecX + vecY + vecZ;
}


void BoundObb::TransformBy( const MatrixTransf& mtx )
{
	m_matCoordSystem.LeftMult( mtx );
	
	if( ! m_matCoordSystem.IsSolidTransform() )
	{
		Vector3f vcCols[3];
		m_matCoordSystem.GetColumns( vcCols[0], vcCols[1], vcCols[2] );

		assert( (vcCols[0] ^ vcCols[1]).GetLength() > 1E-5 );
		assert( (vcCols[0] ^ vcCols[2]).GetLength() > 1E-5 );
		assert( (vcCols[1] ^ vcCols[2]).GetLength() > 1E-5 );

		vcCols[0] *= float(m_vcHalfSize.x);
		vcCols[1] *= float(m_vcHalfSize.y);
		vcCols[2] *= float(m_vcHalfSize.z);

		// Shear -> toute la base est a changer
		if( m_matCoordSystem.HasShear() )
		{
			double adLengths[3];
			adLengths[0] = vcCols[0].GetSquareLength();
			adLengths[1] = vcCols[1].GetSquareLength();
			adLengths[2] = vcCols[2].GetSquareLength();
			
			int32 nIdxFirst = 0;
			int32 nIdxSecond = 1;
			int32 nIdxThird = 2;
			
			if( adLengths[nIdxFirst] < adLengths[nIdxSecond] ) 
				std::swap( nIdxFirst, nIdxSecond );

			if( adLengths[ nIdxSecond ] < adLengths[nIdxThird] ) 
				std::swap( nIdxSecond, nIdxThird );

			if( adLengths[ nIdxFirst ] < adLengths[ nIdxSecond ] ) 
				std::swap( nIdxFirst, nIdxSecond );

			Vector3f vcColX = Normalized( vcCols[nIdxFirst] );
			Vector3f vcColY = Normalized( vcColX ^ vcCols[nIdxSecond]  );
			Vector3f vcColZ = Normalized( vcColX ^ vcColY );

			LengthType lFactX =  LengthType( fabs( vcColX * vcCols[nIdxFirst] ) 
			                               + fabs( vcColX * vcCols[nIdxSecond] )
			                               + fabs( vcColX * vcCols[nIdxThird] ) );

			LengthType lFactY =  LengthType( fabs( vcColY * vcCols[nIdxFirst] ) 
			                               + fabs( vcColY * vcCols[nIdxSecond] )
			                               + fabs( vcColY * vcCols[nIdxThird] ) );

			LengthType lFactZ =  LengthType( fabs( vcColZ * vcCols[nIdxFirst] ) 
			                               + fabs( vcColZ * vcCols[nIdxSecond] )
			                               + fabs( vcColZ * vcCols[nIdxThird] ) );

			m_vcHalfSize = Vector3D( lFactX, lFactY, lFactZ );
			m_matCoordSystem.SetColumns( vcColX, vcColY, vcColZ );
		}
		else
		{
			// Juste du scale -> les transferer dans les m_vcHalfSize
			m_vcHalfSize.x = vcCols[0].GetLength();
			m_vcHalfSize.y = vcCols[1].GetLength();
			m_vcHalfSize.z = vcCols[2].GetLength();
			vcCols[0].Normalize();
			vcCols[1].Normalize();
			vcCols[2].Normalize();

			// Revient a un repere direct si il y a lieu
			if( (vcCols[0] ^ vcCols[1]) * vcCols[2] < 0.0 )
				vcCols[0] *= -1.f;

			m_matCoordSystem.SetColumns( vcCols[0], vcCols[1], vcCols[2] );
		}
	}
}



} // namespaces






